/*
 * Reza Naghibi
 * Weather Channel 2011
 * 
 */

import java.io.BufferedReader;
import java.io.FileReader;

class DTreeClient
{
    //dtree dtnode
    static class DTree
    {
        public char data='\0';
        public int flags=0;
        public DTree[] nodes=new DTree[36];
    }
    
    //logging
    private static final int DTREE_LOG_DTREE_GENERIC=(1<<0);
    private static final int DTREE_LOG_DTREE_INIT=(1<<1);
    private static final int DTREE_LOG_DTREE_EXCEPTION=(1<<2);
    private static final int DTREE_LOG_DTREE_CLASSIFY=(1<<3);
    private static final int DTREE_LOG_UALOOKUP=(1<<4);
    private static final int DTREE_LOG_WALKS=(1<<5);
    //private static final int DTREE_LOG_LEVEL=DTREE_LOG_DTREE_CLASSIFY|DTREE_LOG_DTREE_GENERIC|DTREE_LOG_DTREE_EXCEPTION;
    private static final int DTREE_LOG_LEVEL=0;
    
    private static final int DTREE_DT_FLAG_TOKEN=(1 << 0);
    private static final int DTREE_DT_FLAG_STRONG=(1 << 1);
    private static final int DTREE_DT_FLAG_WEAK=(1 << 2);
    private static final int DTREE_DT_FLAG_NONE=(1 << 3);
    private static final int DTREE_DT_DC_START=4;
    private static final int DTREE_DT_MIN_TLEN=6;
    
    private DTree HEAD=new DTree();
    private int HEAD_count=0;
    private String[] dclass=new String[31-DTREE_DT_DC_START];
    private int dclass_count=0;
    
    public static void main(String args[])
    {
        long start,end;
        long memoryStart=getMemoryUsage(),memory;
        String loadFile="../dtrees/wurfl.dtree";
        String parameter=null;
        
        System.out.println("WURFL DTree Java client");
        
        for(int i=0;i<args.length;i++)
        {
            //-l [dtree file] on command line
            if(args[i].equals("-l") && args.length>(++i))
                loadFile=args[i];
            else
                parameter=args[i];
        }
        
        System.out.println("Loading dtree: '"+loadFile+"'");
        
        DTreeClient client=new DTreeClient();
        
        start=System.nanoTime();
        
        int tokens=client.loadDTree(loadFile);
        
        end=System.nanoTime()-start;
        memory=getMemoryUsage()-memoryStart;
        
        System.out.println("load dtree tokens: "+tokens+" time: "+getTime(end));
        System.out.println("dtree stats: nodes: "+client.getNodeCount()+" mem: "+memory+" ??? bytes");
        
        start=System.nanoTime();
        int nodes=client.walkDTree();
        end=System.nanoTime()-start;
        
        System.out.println("walk tree: "+nodes+" tokens "+client.getNodeCount()+" nodes time: "+getTime(end));
        
        start=System.nanoTime();
        String cls=client.classify("Mozilla/5.0 (Linux; U; Android 2.2; en; HTC Aria A6380 Build/ERE27) AppleWebKit/540.13+ (KHTML, like Gecko) Version/3.1 Mobile Safari/524.15.0".toCharArray());
        end=System.nanoTime()-start;
        
        System.out.println("HTC Aria UA lookup: '"+cls+"' time: "+getTime(end));
        
        if(parameter!=null)
        {
            try
            {
                BufferedReader in=new BufferedReader(new FileReader(parameter));
                System.out.println("UA file: '"+parameter+"'");
                String line;
                long total=0;
                int count=0;
                while((line=in.readLine())!=null)
                {
                    if((DTREE_LOG_LEVEL & DTREE_LOG_UALOOKUP)>0)
                        System.out.println("UA: '"+line+"'");
                    char lineca[]=line.toCharArray();
                    start=System.nanoTime();
                    cls=client.classify(lineca);
                    end=System.nanoTime()-start;
                    total+=end;
                    count++;
                    if((DTREE_LOG_LEVEL & DTREE_LOG_UALOOKUP)>0)
                        System.out.println("UA lookup "+count+": '"+cls+"' time: "+getTime(end));
                }
                total/=count;
                System.out.println("TOTAL average time: "+count+" lookups, "+getTime(total));
                in.close();
            }
            catch(Exception ex)
            {
                System.out.println("UA: '"+parameter+"'");
                start=System.nanoTime();
                cls=client.classify(parameter.toCharArray());
                end=System.nanoTime()-start;

                System.out.println("Param UA lookup: '"+cls+"' time: "+getTime(end));
            }
            
        }
    }
    
    private static String getTime(long diff)
    {
        StringBuilder sb=new StringBuilder("");
        
        sb.append(diff/(1000*1000*1000)).append("s ");
        sb.append(diff/(1000*1000)%1000).append("ms ");
        sb.append(diff/1000%1000).append("us ");
        sb.append(diff%1000).append("ns");
        
        return sb.toString();
    }
    
    private static long getMemoryUsage()
    {
        System.runFinalization();
        System.gc();
        return Runtime.getRuntime().totalMemory()-Runtime.getRuntime().freeMemory();
    }
    
    public int loadDTree(String file)
    {
        int count=0;
        try
        {
            BufferedReader in=new BufferedReader(new FileReader(file));
            String line;
            while((line=in.readLine())!=null)
            {
                if(line.startsWith("#!"))
                {
                    for(String dc:line.substring(2).split(","))
                        getDClassPosition(dc);
                    continue;
                }
                if(line.startsWith("#"))
                    continue;
                String[] tokens=line.split(";");
                if(tokens.length<2)
                    continue;
                String[] attrs=tokens[1].split(",");
                if(DTREE_LOG_LEVEL>0)
                    print(DTREE_LOG_DTREE_INIT,"Parsed token: "+tokens[0]+" attrs: "+attrs.length);
                
                if(addToken(tokens[0],attrs))
                    count++;
                    
                if(((DTREE_LOG_LEVEL & DTREE_LOG_WALKS)>0) && count%10==0)
                {
                    long high=0,low=0,total=0,start,end,nodes=0;
                    for(int i=0;i<5;i++)
                    {
                        start=System.nanoTime();
                        nodes=walkDTree();
                        end=System.nanoTime()-start;
                        if(end==0)
                        {
                            i--;
                            continue;
                        }
                        if(end>high)
                            high=end;
                        if(i==0 || end<low)
                            low=end;
                        total+=end;
                    }
                    end=(total-high-low)/3;
                    System.out.println("walk tree: "+nodes+" tokens "+getNodeCount()+" nodes time: "+getTime(end));
                }
            }
            
            in.close();
        }
        catch(Exception ex)
        {
            if(DTREE_LOG_LEVEL>0)
                print(DTREE_LOG_DTREE_GENERIC,"Exception: "+ex.toString(),ex);
            free();
            return -1;
        }
        
        return count;
    }
    
    private boolean addToken(String token,String attrs[])
    {
        //add to tree
        DTree base=HEAD;
        DTree next=null;
        for(int i=0;i<token.length();i++)
        {
            char c=token.charAt(i);
            int hash=hashChar(c);
            next=base.nodes[hash];
            
            //check for valid char
            if(hash==0 && c!='0')
                return false;

            //empty node
            if(next==null)
            {
                next=base.nodes[hash]=new DTree();
                next.data=c;
                HEAD_count++;
                if(DTREE_LOG_LEVEL>0)
                    print(DTREE_LOG_DTREE_INIT,"ADD: added new node: '"+c+"' level: "+i+" hash: "+hash);
            }

            //end of token
            if(i+1==token.length())
            {
                if(DTREE_LOG_LEVEL>0)
                    print(DTREE_LOG_DTREE_INIT,"ADD: EOT");
                next.flags|=DTREE_DT_FLAG_TOKEN;
                for(int j=0;j<attrs.length;j++)
                {
                    String[] attr=attrs[j].split("=");
                    if(attr.length<2)
                        continue;

                    int dtpos=getDClassPosition(attr[0]);
                    next.flags|=(1 << (dtpos+DTREE_DT_DC_START));

                    if(attr[1].equals("S"))
                        next.flags|=DTREE_DT_FLAG_STRONG;
                    else if(attr[1].equals("W"))
                        next.flags|=DTREE_DT_FLAG_WEAK;
                    else
                        next.flags|=DTREE_DT_FLAG_NONE;
                }
            }

            //traverse
            base=next;
        }
        
        return true;
    }
    
    private int hashChar(char c)
    {
        if(c<='z' && c>='a')
            return c-'a'+10;
        else if(c>='A' && c<='Z')
            return c-'A'+10;
        else if(c<='9' && c>='0')
            return c-'0';
        else
            return 0;
    }
    
    public String classify(char[] ua)
    {
        char p;
        int token=-1;
        int on=0;
        int size=0;
        int valid;
        int flag;
        int wflag=0;
        int nflag=0;
        
        for(int i=0;i<ua.length;i++)
        {
            p=ua[i];
            valid=0;
            if((p>='a' && p<='z') || (p>='A' && p<='Z') || (p>='0' && p<='9'))
            {
                //new token found
                if(on==0)
                {
                    token=i;
                    size=0;
                }
                on=1;
                valid=1;
                size++;
            }
            if(valid==0 || i+1==ua.length)
            {
                //end of token found
                if(on==1 && size>=3)
                {
                    flag=getDTreeFlag(ua,token,size);
                    if(DTREE_LOG_LEVEL>0)
                        print(DTREE_LOG_DTREE_CLASSIFY,"classify() token pos: "+token+" ("+ua[token]+") size: "+size+" flag: "+flag);
                    if((flag&DTREE_DT_FLAG_TOKEN)>0)
                    {
                        if((flag&DTREE_DT_FLAG_STRONG)>0)
                            return getDClass(flag);
                        else if((flag&DTREE_DT_FLAG_WEAK)>0)
                            wflag|=flag;
                        else if((flag&DTREE_DT_FLAG_NONE)>0)
                            nflag|=flag;
                    }
                }
                on=0;
            }
        }

        if(wflag>0)
            return getDClass(wflag);
        
        return getDClass(nflag);
    }
    
    private int getDTreeFlag(char[] ua, int sToken, int size)
    {
        char p;
        char n;
        int start=sToken;
        int lflag=0;
        DTree base;

        base=HEAD.nodes[hashChar(ua[sToken])];

        for(;sToken<ua.length && size>0;size--)
        {
            p=ua[sToken];
            if(base==null)
                break;
            if(base.data!=p && base.data!=(p+'a'-'A'))
                break;

            sToken++;
            
            //exact match
            if(sToken==ua.length || size==1)
            {
                if(base.flags>0)
                    return base.flags;
                else
                    break;
            }

            if((base.flags&DTREE_DT_FLAG_TOKEN)>0 && (sToken-start)>=DTREE_DT_MIN_TLEN)
                lflag=base.flags;
            
            base=base.nodes[hashChar(ua[sToken])];
        }

        return lflag;
    }
    
    private String getDClass(int flag)
    {
        for(int i=0;i<dclass_count;i++)
        {
            if((flag & (1 << (i+DTREE_DT_DC_START)))>0)
                return dclass[i];
        }
        
        return dclass[0];
    }
    
    private int getDClassPosition(String dc)
    {
        for(int i=0;i<dclass_count;i++)
        {
            if(dclass[i].equals(dc))
                return i;
        }
        
        if(dclass_count>=dclass.length)
            return 0;
        
        dclass[dclass_count]=dc;
        dclass_count++;
        
        return dclass_count-1;
    }
    
    public int getNodeCount()
    {
        return HEAD_count;
    }
    
    public void free()
    {
        HEAD=new DTree();
        HEAD_count=0;
        dclass=new String[31-DTREE_DT_DC_START];
        dclass_count=0;
    }
    
    public int walkDTree()
    {
        int nodes=0;
        
        for(int i=0;i<HEAD.nodes.length;i++)
        {
            if(HEAD.nodes[i]!=null)
                nodes+=walkDTree(HEAD.nodes[i],DTREE_LOG_LEVEL>0?new char[256]:null,0);
        }
        
        if(DTREE_LOG_LEVEL>0)
                print(DTREE_LOG_DTREE_GENERIC,"WALK: walked "+nodes+" tokens");
        
        return nodes;
    }
    
    private int walkDTree(DTree node,char[] path,int depth)
    {
        int nodes=0;
        
        if(path!=null)
        {
            path[depth]=node.data;
            path[depth+1]='\0';
        }
        
        if((node.flags&DTREE_DT_FLAG_TOKEN)>0)
        {
            if(DTREE_LOG_LEVEL>0 && path!=null)
                print(DTREE_LOG_DTREE_INIT,"PRINT: found token: "+new String(path)+" flags: "+node.flags);
            nodes++;
        }
        
        for(int i=0;i<node.nodes.length;i++)
        {
            if(node.nodes[i]!=null)
                nodes+=walkDTree(node.nodes[i],path,depth+1);
        }
        
        if(path!=null)
            path[depth]='\0';
        
        return nodes;
    }
    
    private void print(int level, String s)
    {
        print(level,s,null);
    }
    
    private void print(int level, String s, Exception ex)
    {
        if((level&DTREE_LOG_LEVEL)>0)
            System.out.println(s);
        if(ex!=null && (DTREE_LOG_LEVEL&DTREE_LOG_DTREE_EXCEPTION)>0)
            ex.printStackTrace();
    }
}
