/*
 * Reza Naghibi
 * Weather Channel 2011
 * 
 */

#include "dtree_client.h"


//structs for parsing dtree file
typedef struct
{
    char *key;
    char *value;
} dtree_kv_pair;

typedef struct
{
    int count;
    char *name;
    dtree_kv_pair vals[DTREE_DT_MAX_FLAGS];    
} dtree_file_token;


//local prototypes
static int add_dtree_token(dtree_dt_head*,dtree_file_token*);
static inline const char *dtree_find_class(const dtree_dt_head*,int);
static inline int dtree_get_dclass_pos(dtree_dt_head*,char*);
static int get_dtree_flag(char*,int,const dtree_dt_head*);
static packed_ptr alloc_dtree_node(dtree_dt_head*);
static long print_dtree_node(const dtree_dt_head*,const dtree_dt_node*,char*,int);
static inline int dtree_hash_char(char);
static int dtree_printd(int,const char* fmt,...);


//parses a dtree file
int load_dtree(dtree_dt_head *h,char *path)
{
    FILE *f;
    char buf[256];
    char *p;
    char *s;
    int lines=0;
    int i;
    int ret;
    dtree_file_token ft;
    
#if DTREE_PERF_WALKING
    struct timespec startn,endn,diffn;
    long total,high,low,val;
#endif
    
    init_dtree_head(h);
    
    dtree_printd(DTREE_PRINT_GENERIC,"Loading '%s'\n",path);
    
    if(!(f=fopen(path,"r")))
    {
        fprintf(stderr,"DTREE: cannot open '%s'\n",path);
        return -1;
    }
    
    //traverse the loadfile
    while((p=fgets(buf,sizeof(buf),f)))
    {
        if(*buf=='#')
        {
            //initialize the dclasses, order is being forced here
            if(*++p=='!')
            {
                for(++p,s=p;*p;p++)
                {
                    if(*p==',' || !*(p+1))
                    {
                        *p='\0';
                        ret=dtree_get_dclass_pos(h,s);
                        dtree_printd(DTREE_PRINT_INITDTREE,"DCLASS FORCE init: %d='%s'\n",ret,s);
                        s=(p+1);
                    }
                }
            }
            continue;
        }
        
        lines++;
        
        //create a file_token struct pointing to the various parts of the line
        ft.name=buf;
        ft.count=0;
        for(p=buf;*p;p++)
        {
            if(!ft.count && *p==';')
            {
                *p='\0';
                ft.vals[ft.count].key=p+1;
                ft.vals[ft.count].value="";
                ft.count=1;
            }
            else if(ft.count>0 && *p=='=')
            {
                *p='\0';
                ft.vals[ft.count-1].value=p+1;
            }
            else if(ft.count>0 && *p==',')
            {
                *p='\0';
                if(ft.count>=DTREE_DT_MAX_FLAGS)
                    break;
                ft.vals[ft.count].key=p+1;
                ft.vals[ft.count].value="";
                ft.count++;
            }
            else if(*p=='\n')
            {
                *p='\0';
                if(ft.vals[ft.count-1].key==p)
                    ft.count--;
            }
        }
        
        //debugging
        dtree_printd(DTREE_PRINT_INITDTREE,"$$$ LOAD: name: %s",ft.name);
        for(i=0;i<ft.count;i++)
            dtree_printd(DTREE_PRINT_INITDTREE,",%s=%s",ft.vals[i].key,ft.vals[i].value);
        dtree_printd(DTREE_PRINT_INITDTREE,"\n");
        
        //add the token
        ret=add_dtree_token(h,&ft);
        
        if(ret<0)
        {
            free_dtree(h);
            fclose(f);
            return -1;
        }
        else if(ret>0)
            dtree_printd(DTREE_PRINT_INITDTREE,"LOAD: Successfully added %s, nodes: %d, size: %ld\n",ft.name,h->node_count,h->size);
        
#if DTREE_PERF_WALKING
        if(!(lines%10))
        {
            total=high=low=0;
            for(i=0;i<5;i++)
            {
                clock_gettime(CLOCK_REALTIME,&startn);
                ret=print_dtree(h);
                clock_gettime(CLOCK_REALTIME,&endn);
                dttimersubn(&endn,&startn,&diffn);
                val=(diffn.tv_sec*1000*1000*1000)+diffn.tv_nsec;
                if(diffn.tv_nsec==0 && diffn.tv_sec==0)
                {
                    i--;
                    continue;
                }
                if(val>high)
                    high=val;
                if(!i || val<low)
                    low=val;
                total+=val;
            }
            total=(total-high-low)/3;
            printf("walk tree: %d tokens %d nodes time: %lds %ldms %ldus %ldns\n",ret,h->node_count,total/(1000*1000*1000),total/1000000%1000,total/1000%1000,total%1000);
        }
#endif
    }
    
    fclose(f);
    
    return lines;
}

//adds a token to the tree
static int add_dtree_token(dtree_dt_head *h,dtree_file_token *ft)
{
    int flag;
    int hash;
    int i;
    char *p;
    char c;
    int level;
    packed_ptr pp;
    dtree_dt_node *next,*base;
    
    base=&h->head;
    
    for(level=0,p=ft->name;*p;p++,level++)
    {
        c=*(p+1);
        hash=dtree_hash_char(*p);
        
        if(!hash && *p!='0')
            return 0;
        
        pp=(packed_ptr)base->nodes[hash];
        next=DTREE_DT_GETPP(h,pp);
        
        if(!pp)
        {
            pp=alloc_dtree_node(h);
            next=DTREE_DT_GETPP(h,pp);
            if(!pp)
            {
                fprintf(stderr,"DTREE: allocation error detected, aborting: %d\n",h->node_count);
                return -1;
            }
            next->data=*p;
            base->nodes[hash]=pp;
            dtree_printd(DTREE_PRINT_INITDTREE,"ADD: new node '%c' created level: %d hash: %d p(%llx) pp(%llx)\n",*p,level,hash,next,pp);
        }
        
        dtree_printd(DTREE_PRINT_INITDTREE,"ADD: current position '%c' level: %d *p: %c p(%llx) pp(%llx)\n",base->data,level,*p,base,pp);

        if(!c)
        {
            dtree_printd(DTREE_PRINT_INITDTREE,"ADD: EOT\n");
            next->flags |= DTREE_DT_FLAG_TOKEN;
            for(i=0;i<ft->count;i++)
            {
                flag=dtree_get_dclass_pos(h,ft->vals[i].key);
                next->flags |= (1 << (flag+DTREE_DT_FLAG_CPOS));

                if(*ft->vals[i].value=='S')
                    next->flags |= DTREE_DT_FLAG_STRONG;
                else if(*ft->vals[i].value=='W')
                    next->flags |= DTREE_DT_FLAG_WEAK;
                else
                    next->flags |= DTREE_DT_FLAG_NONE;
            }

            //done
            break;
        }
        
        base=next;
    }

    return 1;
}

//hashes a character, on error 0 is returned
static inline int dtree_hash_char(char c)
{
    if(c<='z' && c>='a')
        return c-'a'+10;
    else if(c>='A' && c<='Z')
        return c-'A'+10;
    else if(c<='9' && c>='0')
        return c-'0';
    else
        return 0;
}

//allocates a new node
static packed_ptr alloc_dtree_node(dtree_dt_head *h)
{
    dtree_dt_node *p;
    packed_ptr pp;

    if(h->node_count>=(h->slab_count*DTREE_DT_SLAB_SIZE))
    {
        if(h->slab_count>=DTREE_DT_MAX_SLABS)
        {
            dtree_printd(DTREE_PRINT_INITDTREE,"*** ALLOC: ERROR: all slabs allocated\n");
            return (packed_ptr)0;
        }
        dtree_printd(DTREE_PRINT_INITDTREE,"*** ALLOC: allocating new SLAB\n");
        p=calloc(DTREE_DT_SLAB_SIZE,sizeof(dtree_dt_node));
        if(!p)
            return (packed_ptr)0;
        h->size+=DTREE_DT_SLAB_SIZE*sizeof(dtree_dt_node);
        h->slabs[h->slab_count]=p;
        h->slab_count++;
        h->slab_pos=0;
    }
    
    //return next open node on slab
    p=h->slabs[h->slab_count-1];
    if(!p)
        return (packed_ptr)0;
    p=&p[h->slab_pos];

    pp=DTREE_DT_GENPP(p,h->slab_count-1,h->slab_pos);
    
    dtree_printd(DTREE_PRINT_INITDTREE,"* ALLOC: node %d off of slab %d p(%llx) pp(%llx)\n",h->slab_pos,h->slab_count,p,pp);
    h->slab_pos++;
    h->node_count++;
    
    //bypass null packed_ptr
    if(!pp)
        return alloc_dtree_node(h);
    
    return pp;
}

//classifies the user_agent
const char *dtree_classify(const dtree_dt_head *h,const char *ua)
{
    char *p;
    char *token="";
    int on=0;
    int size=0;
    int valid;
    int flag;
    int wflag=0;
    int nflag=0;
    
    if(!ua)
        return dtree_find_class(h,0);
    
    dtree_printd(DTREE_PRINT_CLASSIFY,"dtree_classify() UA: '%s'\n",ua);
    
    for(p=(char*)ua;*p;p++)
    {
        valid=0;
        if((*p>='a' && *p<='z') || (*p>='A' && *p<='Z') || (*p>='0' && *p<='9'))
        {
            //new token found
            if(!on)
            {
                token=p;
                size=0;
            }
            on=1;
            valid=1;
            size++;
        }
        if(!valid || (!*(p+1)))
        {
            //EOT found
            if(on && size>=3)
            {
                flag=get_dtree_flag(token,size,h);
                dtree_printd(DTREE_PRINT_CLASSIFY,"dtree_classify() token(%d):'%s' = %d\n",size,token,flag);
                if(flag & DTREE_DT_FLAG_TOKEN)
                {
                    if(flag & DTREE_DT_FLAG_STRONG)
                        return dtree_find_class(h,flag);
                    else if(flag & DTREE_DT_FLAG_WEAK)
                        wflag |= flag;
                    else if(flag & DTREE_DT_FLAG_NONE)
                        nflag |= flag;
                }
            }
            on=0;
        }
    }
    
    if(wflag)
        return dtree_find_class(h,wflag);
    
    return dtree_find_class(h,nflag);
}

//gets the flag for a token
static int get_dtree_flag(char *t,int size,const dtree_dt_head *h)
{
    char *p;
    char n;
    int lflag=0;
    packed_ptr pp=(packed_ptr)-1;
    dtree_dt_node *base;
    
    pp=h->head.nodes[dtree_hash_char(*t)];
    base=DTREE_DT_GETPP(h,pp);
    
    for(p=t;size && *p && pp;p++)
    {
        if(base->data!=*p && base->data!=(*p|0x20))
            break;
        
        n=*(p+1);
        
        //exact match
        if(!n || size==1)
        {
            if(base->flags)
                return base->flags;
            else
                break;
        }
        
        //partial match
        if((base->flags & DTREE_DT_FLAG_TOKEN) && (p+1-t)>=DTREE_DT_MIN_TLEN)
            lflag=base->flags;
        
        pp=base->nodes[dtree_hash_char(n)];
        base=DTREE_DT_GETPP(h,pp);
        
        size--;
    }
    
    return lflag;
}

//returns the matching string from flag
static inline const char *dtree_find_class(const dtree_dt_head *h,int flag)
{
    int i;
    
    for(i=0;i<h->dclass_count;i++)
    {
        if(flag & (1 << (i+DTREE_DT_FLAG_CPOS)))
            return h->dclasses[i];
    }
    
    return h->dclasses[0];
}

//gets the dClass position, adds it if it doesnt exist
static inline int dtree_get_dclass_pos(dtree_dt_head *h,char *dclass)
{
    int i;
    
    for(i=0;i<h->dclass_count;i++)
    {
        if(!strncmp(h->dclasses[i],dclass,DTREE_DT_DCLEN))
            return i;
    }
    
    //on error return the first dclass
    if(i>=DTREE_DT_MAX_FLAGS || !*dclass)
        return 0;
    
    //add the dclass
    dtree_printd(DTREE_PRINT_INITDTREE,"DCLASS: dclass init: %d='%s'\n",i,dclass);
    strncpy(h->dclasses[i],dclass,DTREE_DT_DCLEN);
    h->dclass_count++;
    
    return i;
}

//inits the head
void init_dtree_head(dtree_dt_head *h)
{
    memset(h,0,sizeof(dtree_dt_head));
}

//frees the tree
void free_dtree(dtree_dt_head *h)
{
    unsigned int i;
    
    for(i=0;i<h->slab_count;i++)
        free(h->slabs[i]);
    
    init_dtree_head(h);
}

//walks the dtree
long print_dtree(const dtree_dt_head *h)
{
    int i;
    char buf[256];
    long tot=0;
    
    memset(buf,0,sizeof(buf));
    
    for(i=0;i<36;i++)
        tot+=print_dtree_node(h,DTREE_DT_GETPP(h,h->head.nodes[i]),buf,0);
    
    dtree_printd(DTREE_PRINT_GENERIC,"WALK: Walked %ld tokens\n",tot);
    
    return tot;
}

//walks the dtree_node
static long print_dtree_node(const dtree_dt_head *h,const dtree_dt_node *n,char *path,int depth)
{
    int i;
    long tot=0;
    
    if(!n || depth>255)
        return 0;
    
    path[depth]=n->data;
    path[depth+1]='\0';
    
    if(n->flags & DTREE_DT_FLAG_TOKEN)
    {
        dtree_printd(DTREE_PRINT_INITDTREE,"PRINT: token found %s FLAGS: %d\n",path,n->flags);
        tot++;
    }
    
    for(i=0;i<36;i++)
    {
        if(n->nodes[i])
            tot+=print_dtree_node(h,DTREE_DT_GETPP(h,n->nodes[i]),path,depth+1);
    }
    
    path[depth]='\0';
    
    return tot;
}

static int dtree_printd(int level,const char* fmt,...)
{
    int ret=0;

#if DTREE_DEBUG_LOGGING
    va_list ap;
    
    if(level & DTREE_PRINT_ENABLED)
    {
        va_start(ap,fmt);
    
        ret=vprintf(fmt,ap);
    
        va_end(ap);
    }
#endif
    
    return ret;
}


//subtracts timers
void dttimersubn(struct timespec *end,struct timespec *start,struct timespec *result)
{
    result->tv_sec=end->tv_sec-start->tv_sec;
    result->tv_nsec=end->tv_nsec-start->tv_nsec;
    if(result->tv_nsec<0)
    {
      result->tv_sec--;
      result->tv_nsec+=(1000)*(1000)*(1000);
    }
}

