/*
 * Reza Naghibi
 * Weather Channel 2011
 * 
 */

#include <ngx_config.h>
#include <ngx_core.h>
#include <ngx_http.h>

#include "dtree_client.h"


//nginx logging
#define DCLASS_DEBUG     0


//settings struct
typedef struct {
    ngx_flag_t     enable;
    ngx_str_t      dtree_loc;
    ngx_str_t      hfield;
    dtree_dt_head  *head;
} ngx_http_dclass_conf_t;


//prototypes
static void* ngx_http_dclass_create_conf(ngx_conf_t*);
static char* ngx_http_dclass_merge_conf(ngx_conf_t*, void*, void*);
static ngx_int_t ngx_http_dclass_add_class_variable(ngx_conf_t*);
static ngx_int_t ngx_http_dclass_class_variable(ngx_http_request_t*, ngx_http_variable_value_t*, uintptr_t);
static void ngx_http_dclass_cleanup(void*);


//config file commands
static ngx_command_t  ngx_http_dclass_commands[] = {

    { ngx_string("dclass"),
      NGX_HTTP_SRV_CONF|NGX_HTTP_LOC_CONF|NGX_CONF_FLAG,
      ngx_conf_set_flag_slot,
      NGX_HTTP_LOC_CONF_OFFSET,
      offsetof(ngx_http_dclass_conf_t, enable),
      NULL },
      
    { ngx_string("dclass_def"),
      NGX_HTTP_SRV_CONF|NGX_HTTP_LOC_CONF|NGX_CONF_TAKE1,
      ngx_conf_set_str_slot,
      NGX_HTTP_LOC_CONF_OFFSET,
      offsetof(ngx_http_dclass_conf_t, dtree_loc),
      NULL },
      
    { ngx_string("dclass_hfield"),
      NGX_HTTP_SRV_CONF|NGX_HTTP_LOC_CONF|NGX_CONF_TAKE1,
      ngx_conf_set_str_slot,
      NGX_HTTP_LOC_CONF_OFFSET,
      offsetof(ngx_http_dclass_conf_t, hfield),
      NULL },

      ngx_null_command
};


//config callback
static ngx_http_module_t  ngx_http_dclass_module_ctx = {
    ngx_http_dclass_add_class_variable,     /* preconfiguration */
    NULL,                                   /* postconfiguration */

    NULL,                                   /* create main configuration */
    NULL,                                   /* init main configuration */

    NULL,                                   /* create server configuration */
    NULL,                                   /* merge server configuration */

    ngx_http_dclass_create_conf,            /* create location configuration */
    ngx_http_dclass_merge_conf              /* merge location configuration */
};


//module callbacks
ngx_module_t  ngx_http_dclass_module = {
    NGX_MODULE_V1,
    &ngx_http_dclass_module_ctx,            /* module context */
    ngx_http_dclass_commands,               /* module directives */
    NGX_HTTP_MODULE,                        /* module type */
    NULL,                                   /* init master */
    NULL,                                   /* init module */
    NULL,                                   /* init process */
    NULL,                                   /* init thread */
    NULL,                                   /* exit thread */
    NULL,                                   /* exit process */
    NULL,                                   /* exit master */
    NGX_MODULE_V1_PADDING
};


//creates an uninitialized config
static void * ngx_http_dclass_create_conf(ngx_conf_t *cf)
{
    ngx_http_dclass_conf_t *conf;
    ngx_pool_cleanup_t *cln;

    conf = ngx_pcalloc(cf->pool, sizeof(ngx_http_dclass_conf_t));
    if (conf == NULL) {
        return NULL;
    }

    conf->enable = NGX_CONF_UNSET;
    
    cln = ngx_pool_cleanup_add(cf->pool, 0);
    if (cln == NULL) {
        return NULL;
    }
    
    cln->handler = ngx_http_dclass_cleanup;
    cln->data = conf;
    
    return conf;
}


//merges config from parent
static char * ngx_http_dclass_merge_conf(ngx_conf_t *cf, void *parent, void *child)
{
    int ret=0;
    ngx_http_dclass_conf_t *prev = parent;
    ngx_http_dclass_conf_t *conf = child;
    
#if DCLASS_DEBUG
    struct timespec startn,endn,diffn;
#endif
        
    ngx_conf_merge_value(conf->enable, prev->enable, 0);
    
    ngx_conf_merge_str_value(conf->hfield, prev->hfield, "");
    
    if(conf->dtree_loc.data)
    {        
        conf->head=ngx_pcalloc(cf->pool,sizeof(dtree_dt_head));

#if DCLASS_DEBUG
        clock_gettime(CLOCK_REALTIME,&startn);
#endif
        
        ret=load_dtree(conf->head,(char*)conf->dtree_loc.data);
        
#if DCLASS_DEBUG
        clock_gettime(CLOCK_REALTIME,&endn);
        dttimersubn(&endn,&startn,&diffn);
        ngx_log_error(NGX_LOG_NOTICE, cf->log, 0, "DTREE '%s' startup: %ls %lms %lus %lns tokens: %d nodes: %d memory: %l bytes",
                conf->dtree_loc.data,diffn.tv_sec,diffn.tv_nsec/1000000,diffn.tv_nsec/1000%1000,
                diffn.tv_nsec%1000,ret,conf->head->node_count,conf->head->size);
#endif
        
        if(ret<0)
            return NGX_CONF_ERROR;
    }
    else if(prev->head)
        conf->head=prev->head;

    return NGX_CONF_OK;
}


//adds the dclass variable to the config
static ngx_int_t ngx_http_dclass_add_class_variable(ngx_conf_t *cf)
{
    ngx_http_variable_t *var;
    ngx_str_t wcv;
    
    ngx_str_set(&wcv, "dclass");
    
    var = ngx_http_add_variable(cf, &wcv, NGX_HTTP_VAR_CHANGEABLE|NGX_HTTP_VAR_NOHASH);
    
    if (var == NULL)
        return NGX_ERROR;

    var->get_handler = ngx_http_dclass_class_variable;

    return NGX_OK;
}

//populated the dclass variable
static ngx_int_t ngx_http_dclass_class_variable(ngx_http_request_t *r, ngx_http_variable_value_t *v, uintptr_t data)
{
    ngx_http_dclass_conf_t  *cf;
    u_char *hfield=NULL,*class;
    ngx_list_part_t *part;
    ngx_table_elt_t *header;
    ngx_uint_t i;
    
#if DCLASS_DEBUG
    struct timespec startn,endn,diffn;
#endif

    cf = ngx_http_get_module_loc_conf(r, ngx_http_dclass_module);
    
    v->valid=1;
    v->escape=0;
    v->no_cacheable=0;
    v->not_found=0;
    
    if(!cf->enable || !cf->head)
    {
        v->data=(u_char*)"error";
        v->len=5;
        return NGX_OK;
    }
    
    if(!*cf->hfield.data)
    {
        if(r->headers_in.user_agent)
            hfield = r->headers_in.user_agent->value.data;
    }
    else
    {
        part = &r->headers_in.headers.part;
        header = part->elts;
        for (i=0;;i++)
        {
            if (i >= part->nelts)
            {
                if (part->next == NULL)
                    break;
                part = part->next;
                header = part->elts;
                i = 0;
            }
            if(!ngx_strcasecmp(cf->hfield.data,header[i].key.data))
            {
                hfield=header[i].value.data;
                break;
            }
        }
    }
    
    if(hfield == NULL)
    {
        v->data=(u_char*)"unknown";
        v->len=7;
        return NGX_OK;
    }
    
#if DCLASS_DEBUG
    clock_gettime(CLOCK_REALTIME,&startn);
#endif
    
    class=(u_char*)dtree_classify(cf->head,(char*)hfield);
    v->data=class;
    v->len=ngx_strlen(class);
    
#if DCLASS_DEBUG
    clock_gettime(CLOCK_REALTIME,&endn);
    dttimersubn(&endn,&startn,&diffn);
    ngx_log_error(NGX_LOG_NOTICE, r->connection->log, 0, "*!* dclass_class_lookup() time: %ls %lms %lus %lns: '%s' => %s",
        diffn.tv_sec,diffn.tv_nsec/1000000,diffn.tv_nsec/1000%1000,diffn.tv_nsec%1000,hfield,v->data);
#endif

    return NGX_OK;
}

//cleanup
static void ngx_http_dclass_cleanup(void *data)
{
    ngx_http_dclass_conf_t *conf;
    
    conf=(ngx_http_dclass_conf_t*)data;
    
    if(conf->head)
        free_dtree(conf->head);
}
